/*
  System-specific functions
  written by alexander yaworsky
  '99
*/
#include <excpt.h>
#include <windows.h>

#include "stdlib.h"
#include "stringlist.h"
#include "executive.h"
#include "errors.h"
#include "paths.h"
#include "switches.h"
#include "toolhelp.h"


typedef void (*TSuspendThread)( STRINGLIST* Result, DWORD Pid, DWORD Tid );
typedef void (*TResumeThread)( STRINGLIST* Result, DWORD Pid, DWORD Tid );
typedef void (*TKillThread)( STRINGLIST* Result, DWORD Pid, DWORD Tid );
typedef void (*TResumeAllThreads)( STRINGLIST* Result );
typedef void (*TForgetAllThreads)( STRINGLIST* Result );
typedef void (*TGetSuspendedThreads)( STRINGLIST* Result );
typedef BOOL (*TCheckThreadId)( STRINGLIST* Result, HANDLE hSnapshot,
                                DWORD Pid, DWORD Tid );
typedef void (*TKillProcessNotification)( DWORD Pid );

static TSuspendThread             pSuspendThread;
static TResumeThread              pResumeThread;
static TKillThread                pKillThread;
static TResumeAllThreads          pResumeAllThreads;
static TForgetAllThreads          pForgetAllThreads;
static TGetSuspendedThreads       pGetSuspendedThreads;
static TCheckThreadId             pCheckThreadId;
static TKillProcessNotification   pKillProcessNotification;


/********** BEGIN  Win95 implementation  *****************************/

#define VMLDRIOCTL_KILLTHREAD       0x00220007
#define VMLDRIOCTL_SUSPENDTHREAD    0x0022000B
#define VMLDRIOCTL_RESUMETHREAD     0x0022000F
#define VMLDRIOCTL_RESUMEALL        0x00220013
#define VMLDRIOCTL_FORGETALL        0x00220017
#define VMLDRIOCTL_GETSUSPENDED     0x0022001B
#define VMLDRIOCTL_REMOVETHREAD     0x0022001F
#define VMLDRIOCTL_GETKEYSTROKES    0x00220023
#define VMLDRIOCTL_SETGETKEYMAP     0x00220027


static DWORD GetObfuscator()
  {
    extern unsigned GetFS18();
#   pragma aux GetFS18 = \
    "mov EAX,FS:[18h]"\
    value [EAX];

    DWORD  *p;
    DWORD  Tid;

    Tid = GetCurrentThreadId();
    p = (DWORD*) (GetFS18() - 0x10);
    if( *p == 6 )
      return ((DWORD) p) ^ Tid;
    else {
      p = (DWORD*) (GetFS18() - 8);
      if( (*p & 0xFFFF) == 7 )
        return ((DWORD) p) ^ Tid;
    }
    return 0;
  }

static DWORD GetRing0ThreadHandle( DWORD Ring3ThreadId )
  {
    DWORD *p;
    DWORD Rc;

    Rc = 0;
    _try {
      p = (DWORD*) (Ring3ThreadId ^ GetObfuscator());
      if( *p == 6 ) Rc = p[ 0x17 ];
      else if( (*p & 0xFFFF) == 7 ) Rc = p[ 0x16 ];
      else Rc = 0;
    }
    _except (EXCEPTION_EXECUTE_HANDLER) {
      ;
    }
    return Rc;
  }

static HANDLE LoadDriver()
  {
    char VxD[ 32 ];

    lstrcpy( VxD, "\\\\.\\" );
    lstrcat( VxD, LoaderFileName );
    return CreateFile( VxD, 0, 0, 0,
                       CREATE_NEW, FILE_FLAG_DELETE_ON_CLOSE, 0 );
  }

static BOOL GetThrProcAddresses( DWORD* GetCurThr, DWORD* TermThr )
  {
    CHAR        K32Path[ MAX_PATH ];
    HINSTANCE   hK32;
    BOOL        Rc;

    Rc = FALSE;
    GetKernel32Path( K32Path );
    hK32 = LoadLibrary( K32Path );
    if( hK32 != NULL && hK32 != INVALID_HANDLE_VALUE ) {
      *GetCurThr = (DWORD) GetProcAddress( hK32, "GetCurrentThread" );
      *TermThr = (DWORD) GetProcAddress( hK32, "TerminateThread" );
      FreeLibrary( hK32 );
      if( *GetCurThr != 0 && *TermThr != 0 ) Rc = TRUE;
    }
    return Rc;
  }

static void W95SuspendThread( STRINGLIST* Result, DWORD Pid, DWORD Tid )
  {
    DWORD   Buf[2];
    HANDLE  DevH;

    Pid = 0;
    Buf[1] = Tid;
    Buf[0] = GetRing0ThreadHandle( Buf[1] );
    if( Buf[0] == 0 )
      SetReply( Result, ERR_API_ERROR, "GetRing0ThreadHandle" );
    else {
      DevH = LoadDriver();
      if( DevH == INVALID_HANDLE_VALUE )
        SetReply( Result, ERR_API_ERROR, "CreateFile (LoadDriver)" );
      else {
        if( DeviceIoControl( DevH, VMLDRIOCTL_SUSPENDTHREAD,
                             Buf, 8, NULL, 0, NULL, NULL ) )
          SetReply( Result, ERR_NO_ERROR, "" );
        else
          SetReply( Result, ERR_API_ERROR, "DeviceIoControl" );
        CloseHandle( DevH );
      }
    }
  }

static void W95ResumeThread( STRINGLIST* Result, DWORD Pid, DWORD Tid )
  {
    DWORD   R0h;
    HANDLE  DevH;

    Pid = 0;
    R0h = GetRing0ThreadHandle( Tid );
    if( R0h == 0 )
      SetReply( Result, ERR_API_ERROR, "GetRing0ThreadHandle" );
    else {
      DevH = LoadDriver();
      if( DevH == INVALID_HANDLE_VALUE )
        SetReply( Result, ERR_API_ERROR, "CreateFile (LoadDriver)" );
      else {
        if( DeviceIoControl( DevH, VMLDRIOCTL_RESUMETHREAD,
                             &R0h, 4, NULL, 0, NULL, NULL ) )
          SetReply( Result, ERR_NO_ERROR, "" );
        else
          SetReply( Result, ERR_API_ERROR, "DeviceIoControl" );
        CloseHandle( DevH );
      }
    }
  }

static void W95KillThread( STRINGLIST* Result, DWORD Pid, DWORD Tid )
  {
    DWORD   IoctlData[3];
    HANDLE  DevH;

    Pid = 0;
    IoctlData[0] = GetRing0ThreadHandle( Tid );
    if( IoctlData[0] == 0 )
      SetReply( Result, ERR_API_ERROR, "GetRing0ThreadHandle" );
    else {
     DevH = LoadDriver();
      if( DevH == INVALID_HANDLE_VALUE )
        SetReply( Result, ERR_API_ERROR, "CreateFile (LoadDriver)" );
      else {
        if( ! GetThrProcAddresses( &IoctlData[1], &IoctlData[2] ) )
          SetReply( Result, ERR_API_ERROR, "LoadLibrary/GetProcAddress" );
        else
          if( DeviceIoControl( DevH, VMLDRIOCTL_KILLTHREAD,
                               IoctlData, 12, NULL, 0, NULL, NULL ) )
            SetReply( Result, ERR_NO_ERROR, "" );
          else
            SetReply( Result, ERR_API_ERROR, "DeviceIoControl" );
        CloseHandle( DevH );
      }
    }
  }

static void W95ResumeAllThreads( STRINGLIST* Result )
  {
    HANDLE  DevH;

    DevH = LoadDriver();
    if( DevH == INVALID_HANDLE_VALUE )
      SetReply( Result, ERR_API_ERROR, "CreateFile (LoadDriver)" );
    else {
      if( DeviceIoControl( DevH, VMLDRIOCTL_RESUMEALL,
                           NULL, 0, NULL, 0, NULL, NULL ) )
        SetReply( Result, ERR_NO_ERROR, "" );
      else
        SetReply( Result, ERR_API_ERROR, "DeviceIoControl" );
      CloseHandle( DevH );
    }
  }

static void W95ForgetAllThreads( STRINGLIST* Result )
  {
    HANDLE  DevH;

    DevH = LoadDriver();
    if( DevH == INVALID_HANDLE_VALUE )
      SetReply( Result, ERR_API_ERROR, "CreateFile (LoadDriver)" );
    else {
      if( DeviceIoControl( DevH, VMLDRIOCTL_FORGETALL,
                           NULL, 0, NULL, 0, NULL, NULL ) )
        SetReply( Result, ERR_NO_ERROR, "" );
      else
        SetReply( Result, ERR_API_ERROR, "DeviceIoControl" );
      CloseHandle( DevH );
    }
  }

static DWORD W95GetPidForTid( HANDLE Shh, DWORD Tid )
  {
    THREADENTRY32   Pe;

    Pe.dwSize = sizeof( Pe );
    if( ! ThThread32First( Shh, &Pe ) ) return 0;
    do {
      if( Tid == Pe.th32ThreadID  ) return Pe.th32OwnerProcessID;
    } while( ThThread32Next( Shh, &Pe ) );
    return 0;
  }

static void W95GetSuspendedThreads( STRINGLIST* Result )
  {
    HANDLE  Shh, DevH;
    DWORD   Buf[ MAX_SUSPENDED_THREADS ];
    DWORD   SuspCount;
    int     i;
    char    Str[ 32 ];

    Shh = ThCreateToolhelp32Snapshot( TH32CS_SNAPTHREAD, 0 );
    if( Shh == INVALID_HANDLE_VALUE ) {
      SetReply( Result, ERR_API_ERROR, "ThCreateToolhelp32Snapshot" );
      return;
    }
    DevH = LoadDriver();
    if( DevH == INVALID_HANDLE_VALUE )
      SetReply( Result, ERR_API_ERROR, "CreateFile (LoadDriver)" );
    else {
      SuspCount = 0;
      if( DeviceIoControl( DevH, VMLDRIOCTL_GETSUSPENDED,
                           NULL, 0, Buf, sizeof( Buf ), &SuspCount, NULL ) ) {
        SetReply( Result, ERR_NO_ERROR, "" );
        for( i = 0; i < SuspCount; i++ ) {
          wsprintf( Str, "%08X %08X",
                    W95GetPidForTid( Shh, Buf[ i ] ), Buf[ i ] );
          AddToStringList( Result, Str );
        }
      }
      else
        SetReply( Result, ERR_API_ERROR, "DeviceIoControl" );
      CloseHandle( DevH );
    }
    ThCloseHandle( Shh );
  }

static BOOL W95CheckThreadId( STRINGLIST* Result, HANDLE Shh,
                              DWORD Pid, DWORD Tid )
  {
    THREADENTRY32   Pe;
    BOOL            Rc = FALSE;

    Pe.dwSize = sizeof( Pe );
    if( ! ThThread32First( Shh, &Pe ) )
      SetReply( Result, ERR_API_ERROR, "ThThread32First" );
    else {
      do {
        if( Tid == Pe.th32ThreadID && Pid == Pe.th32OwnerProcessID ) {
          Rc = TRUE;
          break;
        }
      } while( ThThread32Next( Shh, &Pe ) );
      if( Rc == FALSE ) {
        SetLastError( ERROR_INVALID_PARAMETER );
        SetReply( Result, ERR_API_ERROR, "OpenThread" );
      }
    }
    return Rc;
  }

static void W95KillProcessNotification( DWORD Pid )
  {
    HANDLE  Shh, DevH;
    DWORD   Buf[ MAX_SUSPENDED_THREADS ];
    DWORD   SuspCount;
    int     i;

    Shh = ThCreateToolhelp32Snapshot( TH32CS_SNAPTHREAD, 0 );
    if( Shh == INVALID_HANDLE_VALUE ) return;
    DevH = LoadDriver();
    if( DevH != INVALID_HANDLE_VALUE ) {
      SuspCount = 0;
      if( DeviceIoControl( DevH, VMLDRIOCTL_GETSUSPENDED,
                           NULL, 0, Buf, sizeof( Buf ), &SuspCount, NULL ) ) {
        for( i = 0; i < SuspCount; i++ )
          if( Pid == W95GetPidForTid( Shh, Buf[ i ] ) )
            DeviceIoControl( DevH, VMLDRIOCTL_REMOVETHREAD,
                                 &Buf[ i ], 4, NULL, 0, NULL, NULL );
      }
      CloseHandle( DevH );
    }
    ThCloseHandle( Shh );
  }

/********** END  Win95 implementation  *****************************/


/********** BEGIN  WinNT implementation  *****************************/

/*** begin - the list of suspended threads *****/

static CRITICAL_SECTION  STCSection;

static DWORD* STList = NULL;
static DWORD  STSize = 0;
static DWORD  STCount = 0;

static void AddToSTList( DWORD Pid, DWORD Tid )
  {
    DWORD*  p;

    if( STCount == STSize ) {
      if( STList == NULL )
        p = (DWORD*) LocalAlloc( LMEM_FIXED,
                                 (STSize + 64) * 2 * sizeof( DWORD ) );
      else
        p = (DWORD*) LocalReAlloc( (HLOCAL) STList,
                                   (STSize + 64) * 2 * sizeof( DWORD ), 0 );
      if( p == NULL ) return;
      STList = p;
      STSize += 64;
    }
    STList[ STCount * 2 ] = Pid;
    STList[ STCount * 2 + 1 ] = Tid;
    STCount++;
  }

static void RemoveFromSTList( DWORD Pid, DWORD Tid )
// if Tid == 0, remove all threads or the given process
// this feature is used by killprocess notification
  {
    DWORD  i;

  Again:
    for( i = 0; i < STCount; i++ )
      if( Tid == 0 ) {
        if( STList[ i * 2 ] == Pid ) break;
      }
      else {
        if( STList[ i * 2 ] == Pid && STList[ i * 2 + 1 ] == Tid ) break;
      }
    if( i >= STCount ) return;
    STCount--;
    if( (STCount - i) > 0 )
      MoveMemory( &STList[ i * 2 ], &STList[ i * 2 + 2 ],
                  (STCount - i) * 2 * sizeof( DWORD ) );
    if( Tid == 0 ) goto Again;
  }

static void ClearSTList()
  {
    EnterCriticalSection( &STCSection );
    STCount = 0;
    LeaveCriticalSection( &STCSection );
  }

static void GetSTList( STRINGLIST* Result )
  {
    DWORD  i;
    char   Buf[ 32 ];

    EnterCriticalSection( &STCSection );
    for( i = 0; i < STCount; i++ ) {
      wsprintf( Buf, "%08X %08X\n", STList[ i * 2 ], STList[ i * 2 + 1 ] );
      AddToStringList( Result, Buf );
    }
    LeaveCriticalSection( &STCSection );
  }

static BOOL CheckItemInSTList(  DWORD Pid, DWORD Tid )
  {
    DWORD  i;

    for( i = 0; i < STCount; i++ )
      if( STList[ i * 2 ] == Pid && STList[ i * 2 + 1 ] == Tid ) return TRUE;
    return FALSE;
  }


/*** end - the list of suspended threads *****/


#pragma pack(0);

typedef struct TagNT_OBJECT_ATTRIBUTES {
    ULONG Length;
    HANDLE RootDirectory;
    VOID* ObjectName;
    ULONG Attributes;
    VOID* SecurityDescriptor;
    VOID* SecurityQualityOfService;
} NT_OBJECT_ATTRIBUTES;

#pragma pack();

typedef DWORD WINAPI (*TNtOpenThread)( HANDLE* ThreadHandle,
                                       DWORD DesiredAccess,
                                       VOID* ObjectAttributes,
                                       LPDWORD ClientId ); // ptr to pid&tid
typedef DWORD WINAPI (*TNtSuspendThread)( HANDLE* ThreadHandle,
                                          LPDWORD SuspendCount );
typedef DWORD WINAPI (*TNtResumeThread)( HANDLE* ThreadHandle,
                                         LPDWORD SuspendCount );
typedef DWORD WINAPI (*TNtTerminateThread)( HANDLE ThreadHandle,
                                            DWORD ExitCode );
typedef DWORD WINAPI (*TNtClose)( HANDLE hObject );

static TNtOpenThread           pNtOpenThread;
static TNtSuspendThread        pNtSuspendThread;
static TNtResumeThread         pNtResumeThread;
static TNtTerminateThread      pNtTerminateThread;
static TNtClose                pNtClose;

static char SD[ SECURITY_DESCRIPTOR_MIN_LENGTH ];


static HANDLE _NTOpenThread( DWORD Pid, DWORD Tid )
  {
    HANDLE  Th;
    DWORD   Rc;
    DWORD   Cid[2];
    NT_OBJECT_ATTRIBUTES  Attr;

    RtlZeroMemory( &Attr, sizeof( Attr ) );
    Attr.Length = sizeof( Attr );
    Attr.SecurityDescriptor = (VOID*) (&SD[0]);
    Cid[0] = Pid;
    Cid[1] = Tid;
    Rc = pNtOpenThread( &Th, THREAD_ALL_ACCESS, &Attr, Cid );
    if( Rc != 0 ) {
      SetLastError( Rc );
      return INVALID_HANDLE_VALUE;
    }
    return Th;
  }

static HANDLE WNTOpenThread( STRINGLIST* Result, DWORD Pid, DWORD Tid )
  {
    HANDLE  Th;

    Th = _NTOpenThread( Pid, Tid );
    if( Th == INVALID_HANDLE_VALUE )
      SetReply( Result, ERR_API_ERROR, "NtOpenThread" );
    return Th;
  }

static void WNTSuspendThread( STRINGLIST* Result, DWORD Pid, DWORD Tid )
  {
    HANDLE  Th;
    DWORD   Rc, Sc;

    EnterCriticalSection( &STCSection );
    if( CheckItemInSTList( Pid, Tid ) == TRUE )
      SetReply( Result, ERR_NO_ERROR, "" );
    else {
      Th = WNTOpenThread( Result, Pid, Tid );
      if( Th != INVALID_HANDLE_VALUE ) {
        Rc = pNtSuspendThread( Th, &Sc );
        if( Rc == 0 ) {
          SetReply( Result, ERR_NO_ERROR, "" );
          AddToSTList( Pid, Tid );
        }
        else {
          SetLastError( Rc );
          SetReply( Result, ERR_API_ERROR, "NtSuspendThread" );
        }
        pNtClose( Th );
      }
    }
    LeaveCriticalSection( &STCSection );
  }

static void WNTResumeThread( STRINGLIST* Result, DWORD Pid, DWORD Tid )
  {
    HANDLE  Th;
    DWORD   Rc, Sc;

    EnterCriticalSection( &STCSection );
    if( CheckItemInSTList( Pid, Tid ) == FALSE )
      SetReply( Result, ERR_NO_ERROR, "" );
    else {
      RemoveFromSTList( Pid, Tid );
      Th = WNTOpenThread( Result, Pid, Tid );
      if( Th != INVALID_HANDLE_VALUE ) {
        Rc = pNtResumeThread( Th, &Sc );
        if( Rc == 0 ) {
          SetReply( Result, ERR_NO_ERROR, "" );
        }
        else {
          SetLastError( Rc );
          SetReply( Result, ERR_API_ERROR, "NtResumeThread" );
        }
        pNtClose( Th );
      }
    }
    LeaveCriticalSection( &STCSection );
  }

static void WNTKillThread( STRINGLIST* Result, DWORD Pid, DWORD Tid )
  {
    HANDLE  Th;
    DWORD   Rc;

    EnterCriticalSection( &STCSection );
    RemoveFromSTList( Pid, Tid );
    LeaveCriticalSection( &STCSection );
    Th = WNTOpenThread( Result, Pid, Tid );
    if( Th != INVALID_HANDLE_VALUE ) {
      Rc = pNtTerminateThread( Th, 0 );
      if( Rc == 0 ) {
        SetReply( Result, ERR_NO_ERROR, "" );
      }
      else {
        SetLastError( Rc );
        SetReply( Result, ERR_API_ERROR, "NtTerminateThread" );
      }
      pNtClose( Th );
    }
  }

static void WNTResumeAllThreads( STRINGLIST* Result )
  {
    HANDLE  Th;
    DWORD   Rc, Sc, i;

    EnterCriticalSection( &STCSection );
    for( i = 0; i < STCount; i++ ) {
      Th = _NTOpenThread( STList[ i * 2 ], STList[ i * 2 + 1 ] );
      if( Th != INVALID_HANDLE_VALUE ) {
        Rc = pNtResumeThread( Th, &Sc );
        pNtClose( Th );
      }
    }
    STCount = 0;
    LeaveCriticalSection( &STCSection );
  }

static void WNTForgetAllThreads( STRINGLIST* Result )
  {
    SetReply( Result, ERR_NO_ERROR, "" );
    ClearSTList( Result );
  }

static void WNTGetSuspendedThreads( STRINGLIST* Result )
  {
    SetReply( Result, ERR_NO_ERROR, "" );
    GetSTList( Result );
  }

static BOOL WNTCheckThreadId( STRINGLIST* Result, HANDLE hSnapshot,
                              DWORD Pid, DWORD Tid )
  {
    Result = NULL; hSnapshot = NULL; Pid = 0; Tid = 0;
    return TRUE;
  }

static void WNTKillProcessNotification( DWORD Pid )
  {
    EnterCriticalSection( &STCSection );
    RemoveFromSTList( Pid, 0 );
    LeaveCriticalSection( &STCSection );
  }

/********** END  WinNT implementation  *****************************/


static volatile LONG Initializing = 0;
static volatile LONG InitCompleted = 0;

static void InitThreadMgr()
  {
    HMODULE  Mh;
    char     Path[ MAX_PATH ];
    LONG     i;

    if( InitCompleted != 0 ) return;
    i = InterlockedExchange( (LPLONG) &Initializing, 1 );
    if( i != 0 ) {
      while( InitCompleted == 0 ) Sleep(0);
      return;
    }
    pSuspendThread = W95SuspendThread;
    pResumeThread = W95ResumeThread;
    pKillThread = W95KillThread;
    pResumeAllThreads = W95ResumeAllThreads;
    pForgetAllThreads = W95ForgetAllThreads;
    pGetSuspendedThreads = W95GetSuspendedThreads;
    pCheckThreadId = W95CheckThreadId;
    pKillProcessNotification = W95KillProcessNotification;
    if( WindowsNT ) {
      GetSystemDirectory( Path, MAX_PATH - lstrlen( NtdllFName ) - 2 );
      if( Path[ lstrlen( Path ) - 1 ] != Slash[0] )
        lstrcat( Path, Slash );
      lstrcat( Path, NtdllFName );
      Mh = GetModuleHandle( Path );
      if( Mh == NULL ) {
        InterlockedIncrement( (LPLONG) &InitCompleted );
        return;
      }
      pNtOpenThread = (TNtOpenThread) GetProcAddress( Mh, "NtOpenThread" );
      pNtSuspendThread = (TNtSuspendThread) GetProcAddress( Mh, "NtSuspendThread" );
      pNtResumeThread = (TNtResumeThread) GetProcAddress( Mh, "NtResumeThread" );
      pNtTerminateThread = (TNtTerminateThread) GetProcAddress( Mh, "NtTerminateThread" );
      pNtClose = (TNtClose) GetProcAddress( Mh, "NtClose" );

      pSuspendThread = WNTSuspendThread;
      pResumeThread = WNTResumeThread;
      pKillThread = WNTKillThread;
      pResumeAllThreads = WNTResumeAllThreads;
      pForgetAllThreads = WNTForgetAllThreads;
      pGetSuspendedThreads = WNTGetSuspendedThreads;
      pCheckThreadId = WNTCheckThreadId;
      pKillProcessNotification = WNTKillProcessNotification;

      InitializeSecurityDescriptor( SD, SECURITY_DESCRIPTOR_REVISION );
      SetSecurityDescriptorDacl( SD, TRUE, (PACL) NULL, FALSE );

      InitializeCriticalSection( &STCSection );
    }
    InterlockedIncrement( (LPLONG) &InitCompleted );
  }


FUNCDEF( Suspend )
  {
    int     i, pid, tid;
    HANDLE  Shh;

    InitThreadMgr();
    if( Params->Count <= 2 ) {
      SetReply( Result, ERR_INVALID_NUM_OF_PARAMS,
                "<process id> <thread id (hex)> [...<thread id (hex)>]" );
      return 1;
    }
    Shh = ThCreateToolhelp32Snapshot( TH32CS_SNAPTHREAD, 0 );
    if( Shh == INVALID_HANDLE_VALUE ) {
      SetReply( Result, ERR_API_ERROR, "ThCreateToolhelp32Snapshot" );
      return 1;
    }
    pid = Strtoul( Params->Value[ 1 ], NULL, 16 );
    for( i = 2; i < Params->Count; i++ ) {
      tid = Strtoul( Params->Value[ i ], NULL, 16 );
      if( pCheckThreadId( Result, Shh, pid, tid ) )
        pSuspendThread( Result, pid, tid );
    }
    ThCloseHandle( Shh );
    return 1;
  }

FUNCDEF( Resume )
  {
    int     i, pid, tid;
    HANDLE  Shh;

    InitThreadMgr();
    if( Params->Count <= 2 ) {
      SetReply( Result, ERR_INVALID_NUM_OF_PARAMS,
                "<process id> <thread id (hex)> [...<thread id (hex)>]" );
      return 1;
    }
    Shh = ThCreateToolhelp32Snapshot( TH32CS_SNAPTHREAD, 0 );
    if( Shh == INVALID_HANDLE_VALUE ) {
      SetReply( Result, ERR_API_ERROR, "ThCreateToolhelp32Snapshot" );
      return 1;
    }
    pid = Strtoul( Params->Value[ 1 ], NULL, 16 );
    for( i = 2; i < Params->Count; i++ ) {
      tid = Strtoul( Params->Value[ i ], NULL, 16 );
      if( pCheckThreadId( Result, Shh, pid, tid ) )
        pResumeThread( Result, pid, tid );
    }
    ThCloseHandle( Shh );
    return 1;
  }

FUNCDEF( KillThread )
  {
    int     i, pid, tid;
    HANDLE  Shh;

    InitThreadMgr();
    if( Params->Count <= 2 ) {
      SetReply( Result, ERR_INVALID_NUM_OF_PARAMS,
                "<process id> <thread id (hex)> [...<thread id (hex)>]" );
      return 1;
    }
    Shh = ThCreateToolhelp32Snapshot( TH32CS_SNAPTHREAD, 0 );
    if( Shh == INVALID_HANDLE_VALUE ) {
      SetReply( Result, ERR_API_ERROR, "ThCreateToolhelp32Snapshot" );
      return 1;
    }
    pid = Strtoul( Params->Value[ 1 ], NULL, 16 );
    for( i = 2; i < Params->Count; i++ ) {
      tid = Strtoul( Params->Value[ i ], NULL, 16 );
      if( pCheckThreadId( Result, Shh, pid, tid ) )
        pKillThread( Result, pid, tid );
    }
    ThCloseHandle( Shh );
    return 1;
  }

FUNCDEF( ResumeAll )
  {
    InitThreadMgr();
    pResumeAllThreads( Result );
    return 1;
  }

FUNCDEF( ForgetAll )
  {
    InitThreadMgr();
    pForgetAllThreads( Result );
    return 1;
  }

FUNCDEF( GetSuspendedThreads )
  {
    InitThreadMgr();
    pGetSuspendedThreads( Result );
    return 1;
  }

void KillProcessNotification( DWORD Pid )
  {
    InitThreadMgr();
    pKillProcessNotification( Pid );
  }

FUNCDEF( ReadCMOS )
  {
    extern BYTE ReadCMOSbyte( BYTE i );
#   pragma aux ReadCMOSbyte = \
    "out  70h,AL"\
    "in   AL,71h"\
    parm [AL] value [AL];

    ULONG   i, cnt, st;
    BYTE    Res[ 256 ];
    char*   Str;

    if( WindowsNT ) {
      SetReply( Result, ERR_NOT_SUPPORTED, "" );
      return 1;
    }
    if( Params->Count <= 2 ) {
      SetReply( Result, ERR_INVALID_NUM_OF_PARAMS, "<starting index> <count>" );
      return 1;
    }
    cnt = Strtoul( Params->Value[2], NULL, 0 );
    st = Strtoul( Params->Value[1], NULL, 0 );
    if( cnt > 0 ) {
      if( cnt > 256 ) cnt = 256;
      for( i = 0; i < cnt; i++ ) Res[i] = ReadCMOSbyte( (BYTE) (st + i) );
      if( (Str = BinaryToStr( Res, cnt )) == NULL ) {
        SetReply( Result, ERR_OUT_OF_MEMORY, "" );
        return 1;
      }
      SetReply( Result, ERR_NO_ERROR, "" );
      AddToStringList( Result, Str );
      LocalFree( Str );
    }
    else
      SetReply( Result, ERR_NO_ERROR, "" );
    return 1;
  }

FUNCDEF( WriteCMOS )
  {
    extern void WriteCMOSbyte( BYTE i, BYTE v );
#   pragma aux WriteCMOSbyte = \
    "out  70h,AL"\
    "mov  AL,AH"\
    "out  71h,AL"\
    parm [AL][AH] modify [EAX];

    ULONG   i, cnt, s;
    BYTE*   Value;

    if( WindowsNT ) {
      SetReply( Result, ERR_NOT_SUPPORTED, "" );
      return 1;
    }
    if( Params->Count <= 3 ) {
      SetReply( Result, ERR_INVALID_NUM_OF_PARAMS,
                "<starting index> <count> <value>" );
      return 1;
    }
    if( (Value = StrToBinary( Params->Value[3] )) == NULL ) {
      SetReply( Result, ERR_OUT_OF_MEMORY, "" );
      return 1;
    }
    cnt = Strtoul( Params->Value[2], NULL, 0 );
    s = Strtoul( Params->Value[1], NULL, 0 );
    if( cnt > 0 ) {
      if( cnt > 256 ) cnt = 256;
      for( i = 0; i < cnt; i++ ) WriteCMOSbyte( (BYTE) (s + i), Value[i] );
    }
    LocalFree( Value );
    SetReply( Result, ERR_NO_ERROR, "" );
    return 1;
  }

static char UnkKey[] = "???";
static char* KeyNames[256] = {
  UnkKey,"ESC","1","2","3","4","5","6","7","8","9","0","-","=","BACKSPACE",
  "TAB","Q","W","E","R","T","Y","U","I","O","P","[","]","ENTER","LCTRL",
  "A","S","D","F","G","H","J","K","L",";","'","~","LSHIFT","\\",
  "Z","X","C","V","B","N","M",",",".","/","RSHIFT","KPAD*","LALT","SPACE",
  "CAPSLOCK","F1","F2","F3","F4","F5","F6","F7","F8","F9","F10","NUMLOCK",
  "SCROLLLOCK","KPAD7","KPAD8","KPAD9","KPAD-","KPAD4","KPAD5","KPAD6","KPAD+",
  "KPAD1","KPAD2","KPAD3","KPAD0","KPAD.",UnkKey,UnkKey,UnkKey,"F11","F12",
  UnkKey,UnkKey,UnkKey,UnkKey,UnkKey,UnkKey,UnkKey,
  UnkKey /*60h*/, UnkKey,UnkKey,UnkKey,UnkKey,UnkKey,UnkKey,UnkKey,
  UnkKey,UnkKey,UnkKey,UnkKey,UnkKey,UnkKey,UnkKey,UnkKey,
  UnkKey /*70h*/, UnkKey,UnkKey,UnkKey,UnkKey,UnkKey,UnkKey,UnkKey,
  UnkKey,UnkKey,UnkKey,UnkKey,UnkKey,UnkKey,UnkKey,UnkKey,

  UnkKey,UnkKey,UnkKey,UnkKey,UnkKey,UnkKey,UnkKey,UnkKey,UnkKey,UnkKey,UnkKey,
  UnkKey,UnkKey,UnkKey,UnkKey,UnkKey,UnkKey,UnkKey,UnkKey,UnkKey,UnkKey,UnkKey,
  UnkKey,UnkKey,UnkKey,UnkKey,UnkKey,UnkKey,"KPADENTER","RCTRL",UnkKey,UnkKey,
  UnkKey,UnkKey,UnkKey,UnkKey,UnkKey,UnkKey,UnkKey,UnkKey,UnkKey,UnkKey,
  "PRINTSCREEN",UnkKey,UnkKey,UnkKey,UnkKey,UnkKey,UnkKey,UnkKey,UnkKey,UnkKey,
  UnkKey,"KPAD/",UnkKey,"SYSREQ","RALT",UnkKey,UnkKey,UnkKey,UnkKey,UnkKey,
  UnkKey,UnkKey,UnkKey,UnkKey,UnkKey,UnkKey,UnkKey,UnkKey,UnkKey,"HOME","UP",
  "PGUP",UnkKey,"LEFT",UnkKey,"RIGHT",UnkKey,"END","DOWN","PGDN","INS","DEL",
  UnkKey,UnkKey,UnkKey,UnkKey,UnkKey, UnkKey,UnkKey,UnkKey,UnkKey,UnkKey,
  UnkKey,UnkKey,
  UnkKey /*60h*/, UnkKey,UnkKey,UnkKey,UnkKey,UnkKey,UnkKey,UnkKey,
  UnkKey,UnkKey,UnkKey,UnkKey,UnkKey,UnkKey,UnkKey,UnkKey,
  UnkKey /*70h*/, UnkKey,UnkKey,UnkKey,UnkKey,UnkKey,UnkKey,UnkKey,
  UnkKey,UnkKey,UnkKey,UnkKey,UnkKey,UnkKey,UnkKey,UnkKey
};


FUNCDEF( Keybuf )
  {
    typedef struct TagSTROKES {
      int      Tail;
      int      NVMark;
      ULONG    KeyCounter;
      int      KeyBufSz;
      unsigned char Keystrokes[4096];
    } STROKES;

    int      First, Count, i;
    char     Msg[ 64 ];
    BOOL     Split;
    STROKES  s;
    HANDLE   DevH;

    /* Get keybstroke data
    */
    if( WindowsNT ) {
      SetReply( Result, ERR_NOT_SUPPORTED, "" );
      return 1;
    }
    else {
      DevH = LoadDriver();
      if( DevH == INVALID_HANDLE_VALUE ) {
        SetReply( Result, ERR_API_ERROR, "CreateFile (LoadDriver)" );
        return 1;
      }
      if( ! DeviceIoControl( DevH, VMLDRIOCTL_GETKEYSTROKES,
                             NULL, 0, &s, sizeof(STROKES), NULL, NULL ) ) {
        SetReply( Result, ERR_API_ERROR, "DeviceIoControl" );
        CloseHandle( DevH );
        return 1;
      }
      CloseHandle( DevH );
    }
    SetReply( Result, ERR_NO_ERROR, "" );
    wsprintf( Msg, "%d keystrokes, NVMark=%d", s.KeyCounter, s.NVMark );
    AddToStringList( Result, Msg );
    First = -1; Count = s.KeyBufSz;
    if( Params->Count > 1 ) Count = Strtoul( Params->Value[1], NULL, 0 );
    if( Params->Count > 2 ) First = Strtoul( Params->Value[2], NULL, 0 );
    if( Count > s.KeyBufSz ) Count = s.KeyBufSz;
    Split = FALSE;
    if( s.Tail >= s.NVMark && s.Keystrokes[ s.Tail ] != 0 ) Split = TRUE;
    if( Split ) {
      if( First < 0 ) {
        // calculate first position if unspecified
        First = s.Tail - Count;
        if( First < s.NVMark ) {
          // wrap arround
          First = s.KeyBufSz - (s.NVMark - First);
          if( First < s.Tail ) {
            // move to nv part
            First = s.NVMark - (s.Tail - First);
            // adjust
            if( First < 0 ) First = 0;
          }
        }
      }
      else {
        if( First >= s.KeyBufSz ) Count = 0;
      }
      for( ; First < s.NVMark && Count > 0; First++, Count-- ) {
        wsprintf( Msg, "%02X %s", (unsigned) s.Keystrokes[ First ],
                  KeyNames[ s.Keystrokes[ First ] ] );
        AddToStringList( Result, Msg );
      }
      if( Count > 0 ) {
        for( First = s.Tail;
             First < s.KeyBufSz && Count > 0; First++, Count-- ) {
          wsprintf( Msg, "%02X %s", (unsigned) s.Keystrokes[ First ],
                    KeyNames[ s.Keystrokes[ First ] ] );
          AddToStringList( Result, Msg );
        }
        if( Count > 0 )
          for( First = s.NVMark;
               First < s.Tail && Count > 0; First++, Count-- ) {
            wsprintf( Msg, "%02X %s", (unsigned) s.Keystrokes[ First ],
                      KeyNames[ s.Keystrokes[ First ] ] );
            AddToStringList( Result, Msg );
          }
      }
    }
    else {
      // buffer is not splitted yet
      if( Count > 0 ) {
        if( First < 0 ) First = s.Tail - Count;
        if( First < 0 ) First = 0;
        if( First > s.Tail ) First = s.Tail;
        if( First + Count > s.Tail ) Count = s.Tail - First;
      }
      if( Count > 0 ) {
        Count += First;
        for( i = First; i < Count; i++ ) {
          wsprintf( Msg, "%02X %s",
                    (unsigned) s.Keystrokes[i], KeyNames[ s.Keystrokes[i] ] );
          AddToStringList( Result, Msg );
        }
      }
    }
    return 1;
  }

FUNCDEF( Remap )
  {
    char    *cp;
    int     i, j;
    char    KeyMap[256], KeyMap2[256];
    HANDLE  DevH;

    /* get current key map
    */
    if( WindowsNT ) {
      SetReply( Result, ERR_NOT_SUPPORTED, "" );
      return 1;
    }
    else {
      DevH = LoadDriver();
      if( DevH == INVALID_HANDLE_VALUE ) {
        SetReply( Result, ERR_API_ERROR, "CreateFile (LoadDriver)" );
        return 1;
      }
      if( ! DeviceIoControl( DevH, VMLDRIOCTL_SETGETKEYMAP,
                             NULL, 0, KeyMap2, 256, NULL, NULL ) ) {
        SetReply( Result, ERR_API_ERROR, "DeviceIoControl" );
        CloseHandle( DevH );
        return 1;
      }
      CloseHandle( DevH );
    }

    if( Params->Count > 1 ) {
      /*
        set entries in key map
      */
      CopyMemory( KeyMap, KeyMap2, 256 );
      for( i = 1; i < Params->Count; i++ ) {
        j = Strtoul( Params->Value[i], &cp, NULL );
        if( j >= 0 && j < 256 )
          for(;;) {
            if( *cp == '\0' ) break;
            do { cp++; } while( *cp == ' ' );
            if( *cp == '\0' ) break;
            KeyMap[ (unsigned char) (j++) ] = (BYTE) Strtoul( cp, &cp, 0 ) & 127;
          }
      }
      /* set/get key map
      */
      if( WindowsNT ) {
        SetReply( Result, ERR_NOT_SUPPORTED, "" );
        return 1;
      }
      else {
        DevH = LoadDriver();
        if( DevH == INVALID_HANDLE_VALUE ) {
          SetReply( Result, ERR_API_ERROR, "CreateFile (LoadDriver)" );
          return 1;
        }
        if( ! DeviceIoControl( DevH, VMLDRIOCTL_SETGETKEYMAP,
                               KeyMap, 256, KeyMap2, 256, NULL, NULL ) ) {
          SetReply( Result, ERR_API_ERROR, "DeviceIoControl" );
          CloseHandle( DevH );
          return 1;
        }
        CloseHandle( DevH );
      }
    }
    SetReply( Result, ERR_NO_ERROR, "" );
    cp = BinaryToStr( KeyMap2, 256 );
    if( cp != NULL ) {
      AddToStringList( Result, cp );
      LocalFree( cp );
    }
    return 1;
  }

FUNCDEF( KeybSave )
  {
    if( WindowsNT ) {
      SetReply( Result, ERR_NOT_SUPPORTED, "" );
      return 1;
    }
    SetReply( Result, ERR_NOT_SUPPORTED, "" );
    return 1;
  }
